<?php

namespace Modules\data\Controllers;

use App\Controllers\BaseController;
use App\Libraries\DatatableQuery;
use App\Traits\CommonTrait;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;
use Modules\data\Models\VulnerableModel;
use Modules\Params\Models\BrancheModel;
use Modules\Params\Models\CollineModel;
use Modules\Params\Models\CommuneModel;
use Modules\Params\Models\TypeAssistanceModel;
use Modules\Params\Models\VulnerabiliteModel;
use Psr\Log\LoggerInterface;
use stdClass;
use UConverter;

use function PHPUnit\Framework\fileExists;

class VulnerableController extends BaseController
{
    public VulnerableModel $vulnerableModel;
    public function initController(RequestInterface $request, ResponseInterface $response, LoggerInterface $logger)
    {
        parent::initController($request, $response, $logger);
        $this->vulnerableModel = new VulnerableModel();
    }

    public function index()
    {
        $this->viewData['url_list'] = "/data/vulnerable/data";
        $this->viewData['js'] = "Common.js";
        $this->viewData['title'] = "Liste des vulnerables";
        $this->viewData['branches'] = CommonTrait::formatData(new BrancheModel());

        return view('data/vulnerable/index', $this->viewData);
    }

    public function add()
    {
        $uri = current_url(true);
        $id = $uri->getSegment(5);

        $this->viewData['url_list'] = "";
        $this->viewData['js'] = "Volontaire.js";
        $this->viewData['route'] = "/data/vulnerable/save";
        $this->viewData['vulnerable'] =  $this->initVulnerable($id);
        $this->viewData['branches'] = CommonTrait::formatData(new BrancheModel());
        $this->viewData['vulnerabilites'] = CommonTrait::formatData(new VulnerabiliteModel());
        $this->viewData['assistances'] = CommonTrait::formatData(new TypeAssistanceModel());
        $this->viewData['categorieAssistances'] = CommonTrait::categorieAssistances();

        if ($id > 0)
            $title = "Editer une vulnérable";
        else
            $title = "Nouvelle vulnérable";

        $this->viewData['title'] = $title;

        return view('data/vulnerable/add', $this->viewData);
    }

    public function save()
    {
        $id = $this->request->getPost('id');

        if (!$this->validate($this->_rules())) {

            $this->viewData['url_list'] = "/data/vulnerable/data";
            $this->viewData['js'] = "";
            $this->viewData['route'] = "/data/vulnerable/save";
            $this->viewData['vulnerable'] =  $this->initVulnerable($id);
            $this->viewData['branches'] = CommonTrait::formatData(new BrancheModel());
            $this->viewData['vulnerabilites'] = CommonTrait::formatData(new VulnerabiliteModel());
            $this->viewData['assistances'] = CommonTrait::formatData(new TypeAssistanceModel());
            $this->viewData['categorieAssistances'] = CommonTrait::categorieAssistances();

            if ($id > 0)
                $title = "Editer une vulnérable";
            else
                $title = "Nouvelle vulnérable";

            $this->viewData['title'] = $title;

            return view('data/vulnerable/add', $this->viewData);
        } else {

            $uneVulnerable = $this->uneVulnerable();
            if ($id) {
                $this->session->setFlashdata('message', "<div class='row alert alert-success'>La vulnérable a été modifiée</div>");
                $stmt = $this->vulnerableModel->update($id, $uneVulnerable);
            } else {
                $this->session->setFlashdata('message', "<div class='row alert alert-success'>La vulnérable a été bien enregistrée</div>");
                $stmt = $this->vulnerableModel->save($uneVulnerable);
            }

            if ($stmt) {
                return redirect()->to(site_url($this->viewData['locale'] . '/data/vulnerable'));
            }
        }
    }


    public function delete()
    {
        $uri = current_url(true);
        $id = $uri->getSegment(6);

        $stmnt = $this->vulnerableModel->delete($id);

        if ($stmnt) {
            $this->session->setFlashdata('message', "<div class='row alert alert-success'>La formation a été suprimée</div>");
            return redirect()->to(site_url($this->viewData['locale'] . '/data/vulnerable'));
        }
    }

    private function uneVulnerable()
    {
        $vulnerable = [];
        $vulnerable['brancheId'] = $this->request->getPost('brancheId');
        $vulnerable['communeId'] = $this->request->getPost('communeId');
        $vulnerable['zoneId'] = $this->request->getPost('zoneId');
        $vulnerable['collineId'] = $this->request->getPost('collineId');
        $vulnerable['categorieAssistanceId'] = $this->request->getPost('categorieAssistanceId');
        $vulnerable['nomPrenomChefMenage'] = $this->request->getPost('nomPrenomChefMenage');
        $vulnerable['age'] = $this->request->getPost('age');
        $vulnerable['tailleMenage'] = $this->request->getPost('tailleMenage');
        $vulnerable['coutEstimatifAssistance'] = $this->request->getPost('coutEstimatifAssistance');
        $vulnerable['typeVulnerabiliteId'] = $this->request->getPost('typeVulnerabiliteId');
        $vulnerable['typeAssistanceId'] = $this->request->getPost('typeAssistanceId');
        $vulnerable['nbPersonneAssisteH'] = $this->request->getPost('nbPersonneAssisteH');
        $vulnerable['nbPersonneAssisteF'] = $this->request->getPost('nbPersonneAssisteF');
        $vulnerable['dateAssistance'] = $this->request->getPost('dateAssistance');
        $vulnerable['userId'] = \auth()->user()->id;

        $file = $this->request->getFile('pathFile');
        if ($file->isValid() && !$file->hasMoved()) {
            $newName = $file->getRandomName();
            $file->move('files/vulnerable', $newName);
            $vulnerable['pathFile'] = $newName;
        }

        return $vulnerable;
    }

    private function initVulnerable($id = NULL)
    {
        $vulnerable = NULL;
        if ($id) {
            $vulnerable = $this->vulnerableModel->find($id);
        } else {
            $vulnerable = new stdClass();
            $vulnerable->id = 0;
            $vulnerable->brancheId = NULL;
            $vulnerable->communeId = NULL;
            $vulnerable->zoneId = NULL;
            $vulnerable->collineId = NULL;
            $vulnerable->nomPrenomChefMenage = '';
            $vulnerable->age = NULL;
            $vulnerable->tailleMenage = 0;
            $vulnerable->typeAssistanceId = 0;
            $vulnerable->coutEstimatifAssistance = 0;
            $vulnerable->typeVulnerabiliteId = 0;
            $vulnerable->categorieAssistanceId = 0;
            $vulnerable->nbPersonneAssisteH = 0;
            $vulnerable->nbPersonneAssisteF = 0;
            $vulnerable->dateAssistance = NULL;
            $vulnerable->pathFile = NULL;
        }

        return $vulnerable;
    }

    private function _rules()
    {
        $rules = [];
        $rules['brancheId'] = ['rules'  => 'required'];
        $rules['communeId'] = ['rules'  => 'required'];
        $rules['zoneId'] = ['rules'  => 'required'];
        $rules['collineId'] = ['rules'  => 'required'];
        $rules['typeAssistanceId'] = ['rules'  => 'required'];
        $rules['nomPrenomChefMenage'] = ['rules'  => 'required'];
        $rules['tailleMenage'] = ['rules'  => 'required'];
        $rules['typeVulnerabiliteId'] = ['rules'  => 'required|is_natural'];
        $rules['categorieAssistanceId'] = ['rules'  => 'required'];

        return $rules;
    }

    function fecth()
    {
        $search_column = ['brancheId', 'communeId', 'zoneId', 'collineId', 'nomPrenomChefMenage', 'age', 'tailleMenage', 'coutEstimatifAssistance', 'typeVulnerabiliteId', 'typeAssistanceId', 'nbPersonneAssisteH', 'nbPersonneAssisteF', 'dateAssistance'];
        $order_column = ['brancheId', 'communeId', 'zoneId', 'collineId', 'nomPrenomChefMenage', 'age', 'tailleMenage', 'coutEstimatifAssistance', 'typeVulnerabiliteId', 'typeAssistanceId', 'nbPersonneAssisteH', 'nbPersonneAssisteF', 'dateAssistance'];

        $vulnerableMdl = $this->vulnerableModel;
        if (\auth()->user()->inGroup('brancheManager')) {
            $vulnerableMdl->where('brancheId', \auth()->user()->brancheId);
        }

        if (\auth()->user()->inGroup('communeManager')) {
            $vulnerableMdl->where('communeId', \auth()->user()->communeId);
        }

        if ($this->request->getPost('brancheId') > 0) {
            $vulnerableMdl->where('brancheId', $this->request->getPost('brancheId'));
        }

        $fetch_data = DatatableQuery::build_query($vulnerableMdl, $search_column, $order_column);
        $output = $this->formatVulnerables($this->viewData['locale'], $fetch_data);

        return $this->response->setJSON($output);
    }

    private function formatVulnerables($locale, $datas)
    {
        $i = 1;
        $array_data = array();
        if (!empty($datas['data'])) {
            foreach ($datas['data'] as $data) {

                $sub_array = array();

                $sub_array[] =  $i;
                $sub_array[] =   $data->brancheId > 0 ? CommonTrait::formatData(new BrancheModel(), $data->brancheId): "";
                $sub_array[] =   $data->communeId > 0 ? CommonTrait::formatData(new CommuneModel(), $data->communeId): "";
                $sub_array[] =  $data->categorieAssistanceId > 0 ? CommonTrait::categorieAssistances($data->categorieAssistanceId) : "";
                $sub_array[] =  $data->nomPrenomChefMenage . " " . $data->age;
                $sub_array[] =  $data->coutEstimatifAssistance;
                $sub_array[] =  $data->typeVulnerabiliteId > 0 ? CommonTrait::formatData(new VulnerabiliteModel(), $data->typeVulnerabiliteId) : "";
                $sub_array[] =  $data->typeAssistanceId > 0 ? CommonTrait::formatData(new TypeAssistanceModel(), $data->typeAssistanceId) : "";
                $sub_array[] =  $data->nbPersonneAssisteH;
                $sub_array[] =  $data->nbPersonneAssisteF;
                $sub_array[] =  $data->dateAssistance;
                $sub_array[] = !empty($data->pathFile) ? anchor(base_url() . "files/vulnerable/" . $data->pathFile, '...'.substr($data->pathFile, -8), "target='_blank'") : "";


                $option = "-";
                if (\auth()->user()->inGroup('hr', 'admin') || (\auth()->user()->hasPermission('params.edit') && \auth()->user()->hasPermission('params.delete'))) {
                    $option = anchor($locale . "/data/vulnerable/add/" . $data->id, "<i class='fa fa-edit'></i>", ['class' => 'text-info']) . " "
                        . anchor($locale . "/data/vulnerable/delete/" . $data->id, "<i class='fa fa-trash'></i>", ['class' => 'text-danger']);
                }
                $sub_array[] = $option;

                $array_data[] = $sub_array;
                $i++;
            }
        }

        return [
            "draw" => intval($_POST["draw"]),
            "recordsTotal" => $datas['recordsTotal'],
            "recordsFiltered" => $datas['recordsFiltered'],
            "data" => $array_data
        ];
    }
    function export()
    {
        $uri = current_url(true);
        $brancheId = $uri->getSegment(5);

        if ($brancheId > 0) {
            $branche = CommonTrait::formatData(new BrancheModel(), $brancheId);
        } else {
            $branche = "Toutes les branches";
        }

        $fileName = str_replace(' ', '', $branche);
        $csv_file_name = "rapportExport/vulnerable/vulnerable_" . $fileName . ".csv";
        $file = fopen($csv_file_name, 'w');

        $arrays = [];
        $arrays[0] = ['', $this->formatText("Vulnerabilite")];
        $arrays[1] = ['', $this->formatText($branche)];
        $arrays[2] = [];
        $arrays[3] = [
            '#',
            $this->formatText('Province'),
            $this->formatText('Branche'),
            $this->formatText('Colline'),
            $this->formatText('Categorie assistance'),
            $this->formatText('Nom et Prénom du chef de ménage'),
            $this->formatText('Age'),
            $this->formatText('Taille du ménage'),
            $this->formatText("Coût estimatif de l'assistance"),
            $this->formatText('Type de Vulnérabilité'),
            $this->formatText("Type d'assistance"),
            $this->formatText("Nombre de personnes assistées (H)"),
            $this->formatText("Nombre de personnes Assistées (F)"),
            $this->formatText("Date d'assistance")
        ];

        if ($brancheId > 0) {
            $vulnerables = $this->vulnerableModel->where('brancheId', $brancheId)->findAll();
        } else {
            $vulnerables = $this->vulnerableModel->findAll();
        }

        
        $i = 1;
        foreach ($vulnerables as $vulnerable) {
            $sub_array = [];
            $sub_array[] = $i;
            $sub_array[] =  $vulnerable->brancheId > 0 ? $this->formatText(CommonTrait::formatData(new BrancheModel(), $vulnerable->brancheId)) : "";
            $sub_array[] =  $vulnerable->communeId > 0 ? $this->formatText(CommonTrait::formatData(new CommuneModel(), $vulnerable->communeId)) : "";
            $sub_array[] =  $vulnerable->collineId > 0 ? $this->formatText(CommonTrait::formatData(new CollineModel(), $vulnerable->collineId)) : "";
            $sub_array[] =  $vulnerable->categorieAssistanceId > 0 ? CommonTrait::categorieAssistances($vulnerable->categorieAssistanceId) : "";
            $sub_array[] =  $this->formatText($vulnerable->nomPrenomChefMenage);
            $sub_array[] =  $vulnerable->age;
            $sub_array[] =  $vulnerable->tailleMenage;
            $sub_array[] =  $vulnerable->coutEstimatifAssistance;
            $sub_array[] =  $vulnerable->typeVulnerabiliteId > 0 ? $this->formatText(CommonTrait::formatData(new VulnerabiliteModel(), $vulnerable->typeVulnerabiliteId)): "";
            $sub_array[] =  $vulnerable->typeAssistanceId > 0 ?$this->formatText(CommonTrait::formatData(new TypeAssistanceModel(), $vulnerable->typeAssistanceId)): "";
            $sub_array[] =  $vulnerable->nbPersonneAssisteH;
            $sub_array[] =  $vulnerable->nbPersonneAssisteF;
            $sub_array[] =  $vulnerable->dateAssistance;

            $arrays[] = $sub_array;
            $i++;
        }


        foreach ($arrays as $array) {
            fputcsv($file, $array,";");
        }
        fclose($file);
        $btn = anchor($csv_file_name, "Exporter un <i class='fa fa-file-csv'></i>", ['class' => 'text-success', 'target' => '_blank']);

        $status = 0;
        if (fileExists($csv_file_name)) {
            $status = 1;
        }

       return $this->response->setJSON(['status' => $status, 'btnExport' => $btn]);
    }

    function formatText($string = "")
    {
        return UConverter::transcode($string, 'ISO-8859-1', 'UTF8');
    }
}
